<?php

declare(strict_types=1);

namespace Erlage\Photogram\Requests\Post\Comment\Like;

use Erlage\Photogram\Settings;
use Erlage\Photogram\Data\Query;
use Erlage\Photogram\Helpers\TraitFeedHelper;
use Erlage\Photogram\Constants\ServerConstants;
use Erlage\Photogram\Data\Models\Post\PostModel;
use Erlage\Photogram\Data\Models\User\UserModel;
use Erlage\Photogram\Data\Tables\User\UserTable;
use Erlage\Photogram\Constants\ResponseConstants;
use Erlage\Photogram\Exceptions\RequestException;
use Erlage\Photogram\Pattern\ExceptionalRequests;
use Erlage\Photogram\Data\Tables\Sys\RequestTable;
use Erlage\Photogram\Data\Models\User\UserModelHelper;
use Erlage\Photogram\Data\Tables\Post\PostCommentLikeTable;
use Erlage\Photogram\Data\Models\Post\Comment\PostCommentModel;
use Erlage\Photogram\Data\Models\Post\Comment\Like\PostCommentLikeModel;

final class PostCommentLikeContent extends ExceptionalRequests
{
    use TraitFeedHelper;

    public static function load(string $loadType): void
    {
        self::feedHelperInit('', $loadType);

        self::process(function ()
        {
            /*
            |--------------------------------------------------------------------------
            | get data from request
            |--------------------------------------------------------------------------
            */

            $postCommentIdFromReq = self::$request -> findKey(
                PostCommentLikeTable::LIKED_POST_COMMENT_ID,
                RequestTable::PAYLOAD,
                PostCommentLikeTable::TABLE_NAME
            );

            $offset = self::$request -> findKeyOffset(PostCommentLikeTable::ID, PostCommentLikeTable::TABLE_NAME);

            self::ensureValue(ResponseConstants::ERROR_BAD_REQUEST_MSG, $postCommentIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | make sure user is authenticated
            |--------------------------------------------------------------------------
            */

            self::userEnsureAuthenticated();

            /*
            |--------------------------------------------------------------------------
            | ensure comment exists
            |--------------------------------------------------------------------------
            */

            $targetPostCommentModel = PostCommentModel::findFromId_throwException($postCommentIdFromReq);

            /*
            |--------------------------------------------------------------------------
            | check if post exists
            |--------------------------------------------------------------------------
            */

            $targetPostModel = PostModel::findFromId_throwException($targetPostCommentModel -> getParentPostId());

            /*
            |--------------------------------------------------------------------------
            | check if post publisher is available
            |--------------------------------------------------------------------------
            */

            $postOwnerUserModel = UserModel::findFromId_throwException($targetPostModel -> getOwnerUserId());

            /*
            |--------------------------------------------------------------------------
            | privacy checks
            |--------------------------------------------------------------------------
            */

            if ( ! UserModelHelper::isUserContentAvailable($postOwnerUserModel, self::$authedUserModel))
            {
                throw new RequestException(ResponseConstants::ERROR_BAD_REQUEST_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | query builder
            |--------------------------------------------------------------------------
            */

            $postCommentLikeTableQuery = (new Query()) -> from(PostCommentLikeTable::TABLE_NAME);

            /*
            |--------------------------------------------------------------------------
            | select where
            |--------------------------------------------------------------------------
            */

            $postCommentLikeTableQuery -> where(PostCommentLikeTable::LIKED_POST_COMMENT_ID, $targetPostCommentModel -> getId());

            $postCommentLikeTableQuery -> where(PostCommentLikeTable::PARENT_POST_ID, $targetPostModel -> getId());

            /*
            |--------------------------------------------------------------------------
            | post likes selection order
            |--------------------------------------------------------------------------
            */

            if (self::isLoadingLatestContent())
            {
                $postCommentLikeTableQuery -> greaterThan(PostCommentLikeTable::ID, $offset);
            }
            else
            {
                $postCommentLikeTableQuery -> lessThan(PostCommentLikeTable::ID, $offset);
            }

            /*
            |--------------------------------------------------------------------------
            | order by post like id & limit
            |--------------------------------------------------------------------------
            */

            $postCommentLikeTableQuery
                -> orderByDesc(PostCommentLikeTable::ID)
                -> limit(Settings::getString(ServerConstants::SS_INT_LIMIT_LOAD_POST_COMMENT_LIKE));

            /*
            |--------------------------------------------------------------------------
            | get post likes beans
            |--------------------------------------------------------------------------
            */

            $postCommentLikeBeans = $postCommentLikeTableQuery -> select();

            /*
            |--------------------------------------------------------------------------
            | check end of results
            |--------------------------------------------------------------------------
            */

            if (0 == \count($postCommentLikeBeans))
            {
                return self::setMessage(ResponseConstants::END_OF_RESULTS_MSG);
            }

            /*
            |--------------------------------------------------------------------------
            | prepare maps
            |--------------------------------------------------------------------------
            */

            self::processBeans(PostCommentLikeTable::getTableName(), $postCommentLikeBeans, function (PostCommentLikeModel $model)
            {
                /*
                |--------------------------------------------------------------------------
                | list of users(who's likes are selected)
                |--------------------------------------------------------------------------
                */

                self::addDependency(UserTable::getTableName(), $model -> getOwnerUserId());
            });
        });
    }
}
